/* FILE: mstation.c                             (D. Tottingham  04/07/91)

This is a collection of C functions that manage the station information for
for xdetect.  All functions have been written and compiled medium model.
The following functions are included:

st_add_station ()               add station to station queue.
st_build_textline ()            build a textline for the mscreen module
st_get_authority ()             get organization processing the data.
st_get_head_station ()          get station at head of station queue
st_get_next_station ()          get next station from station queue
st_get_netwname ()              get network name
st_get_station ()               get station from station queue
st_initialize_params ()         initialize key parameters
st_initialize_stations ()       initialize station list
st_set_authority ()             set authority code
st_set_gain ()                  set gain for particular channel
st_set_netwname ()              set network name

EXTERNAL FUNCTIONS CALLED:

er_abort ()                     display an error message then quit
dt_get_channel_gain ()          get channel gain
dt_get_clip_value ()            get clip value
dt_get_data_type ()             get data type
dt_get_data_units ()            get data units
dt_get_dc_offset ()             get dc offset
dt_get_input_range ()           get input range
ll_fold ()                      convert lat, lon to geocentric lat, lon
ll_to_xy ()                     convert lat, lon to x, y
q_enqueue ()                    enqueue a data link on a data queue
q_initialize ()                 initialize a data queue
suds_initialize ()              initialize a suds structure
suds_initialize_tag ()          initialize a suds structtag
t_set_netwname ()               set network name
u_strncpy ()                    copies far string2 to far string1
u_strnlen ()                    get length of far string
u_timestamp ()                  get timestamp

HISTORY:
   none

*/


/*************************************************************************
                            INCLUDE FILES

*************************************************************************/
#include <io.h>
#include <malloc.h>
#include <math.h>
#include <stdio.h>
#include <string.h>

#include "mconst.h"
#include "mdosfunc.h"
#include "mdt28xx.h"
#include "merror.h"
#include "mlatlon.h"
#include "mqueue.h"
#include "mstation.h"
#include "msudsini.h"
#include "mtrigger.h"
#include "mutils.h"


/*************************************************************************
                                GLOBALS

*************************************************************************/
PRIVATE Q_QUEUE station_queue;
PRIVATE Q_LINK * head_ptr;
PRIVATE LL_LATLON mean_latlon;
PRIVATE double mean_lat, mean_lon;
PRIVATE char netwname[4];
PRIVATE unsigned int n_stations, station_fd, station_size;
PRIVATE int authority;


/*=======================================================================*
 *                               b_search                                *
 *=======================================================================*/
/* Look in sorted station file for a specific station using a hash value
   based binary search.                                                  */

PRIVATE
int b_search (fd, size, search_key, stationcomp_ptr)
unsigned int fd;
unsigned int size;
unsigned long search_key;
SUDS_STATIONCOMP far * stationcomp_ptr;
{
   static SUDS_STRUCTTAG far structtag;
   unsigned int head, tail, current;
   unsigned long current_key;

   head = 0, tail = size - 1;
   while (1) {
      current = (head + tail) / 2;

      /* Get the current station component from station file */
      seek_file (fd, (long) (current * (sizeof(SUDS_STRUCTTAG) +
                             sizeof(SUDS_STATIONCOMP))), SEEK_SET);
      read_file (fd, sizeof(SUDS_STRUCTTAG), ((int far *)&structtag));
      if (structtag.id_struct != STATIONCOMP) er_abort (ST_INVALID_FILE);
      read_file (fd, sizeof(SUDS_STATIONCOMP), ((int far *)stationcomp_ptr));

      /* Do the search */
      current_key = *((unsigned long far *) &(stationcomp_ptr->channel));
/* printf ("search_key, current_key = %lu, %lu\n", search_key, current_key); */
      if (search_key == current_key)
         return (current);
      else if (head == tail)
         return (-1);
      else if (search_key < current_key)
         tail = (tail == current) ? head : current;
      else {
         head = (head == current) ? tail : current;
/* printf ("head, current = %u, %u\n", head, current);  */
      }
   }
}

/*=======================================================================*
 *                            get_hashvalue                              *
 *=======================================================================*/
/* Generate an aggregate hash value for given network, station name, and
   component.                                                            */

PRIVATE
unsigned long get_hashvalue (network, st_name, component)
char * network;
char * st_name;
char component;
{
   static unsigned int shifter1[] = {25, 16, 23, 18};
   static unsigned int shifter2[] = {12, 0, 5, 7, 2};
   static unsigned int shifter3 = 10;
   unsigned long hash;
   unsigned int i;

   hash = 0;
   for (i = 0; network[i] && i < 4; i++)
      hash += ((unsigned long) ((network[i] - 48) & 0x7f)) << shifter1[i];

   for (i = 0; st_name[i] && i < 5; i++)
      hash += ((unsigned long) ((st_name[i] - 48) & 0x7f)) << shifter2[i];

   hash += ((unsigned long) component) << shifter3;
   return (hash);
}

/*=======================================================================*
 *                            in_masterlist                              *
 *=======================================================================*/
/* Determines if station component is defined in STATION.@@@.  If it is,
   the station information is copied directly into the suds structure.   */

PRIVATE
FLAG in_masterlist (network, st_name, component, ptr)
char network[];
char st_name[];
char component;
SUDS_STATIONCOMP far * ptr;
{
   static SUDS_STRUCTTAG far structtag;
   unsigned int size;
   unsigned long hash_value;
   int current_ptr, i;
   FLAG done, first_time;

   hash_value = get_hashvalue (network, st_name, component);
   if ((current_ptr = b_search (station_fd, station_size, hash_value, ptr)) >= 0) {
/* printf ("Found hash value for ");
for (i = 0; i < 10; i++)
   printf ("%c", ((char far *)ptr->sc_name.network)[i]);
printf ("\n");  */
      /* Is this the right station ? */
      if (!u_strncmp (ptr->sc_name.network, network, 4) &&
          !u_strncmp (ptr->sc_name.st_name, st_name, 5) &&
          ptr->sc_name.component == component) {
/* printf ("Found %s in station file\n", st_name);  */
         return (TRUE);
      }

      /* Wrong station. Check for collisions.
         Stations with unique hash values have their data_type = 0.
         All stations with the same hash value receive incremental data_type
         values (i.e. 0, 1, 2 ...) as they are added to the station file.
         To search a group of stations with the same hash value for a
         specific station, we simply scan between stations with data_type
         values = 0.                                                      */

      if (ptr->data_type) {
         current_ptr -= ptr->data_type;
/* printf ("Adjusted current_ptr by %d\n", ptr->data_type);  */
      } else current_ptr++;

      for (first_time = TRUE, done = FALSE; ! done;) {
         seek_file (station_fd, (long) (current_ptr * (sizeof(SUDS_STRUCTTAG) +
                                        sizeof(SUDS_STATIONCOMP))), SEEK_SET);
         read_file (station_fd, sizeof(SUDS_STRUCTTAG), ((int far *)&structtag));
         if (structtag.id_struct != STATIONCOMP) er_abort (ST_INVALID_FILE);
         read_file (station_fd, sizeof(SUDS_STATIONCOMP), ((int far *)ptr));
         if (ptr->data_type || first_time) {
            if (!u_strncmp (ptr->sc_name.network, network, 4) &&
                !u_strncmp (ptr->sc_name.st_name, st_name, 5) &&
                ptr->sc_name.component == component) {
/* printf ("Found %s in station file\n", st_name);  */
               return (TRUE);
            }
            first_time = FALSE;
         } else done = TRUE;
         current_ptr++;
      }
   }
/* printf ("Could not find %s in station file: hash value = %lu\n", st_name,
        hash_value);   */
   return (FALSE);
}

/*=======================================================================*
 *                           st_add_station                              *
 *=======================================================================*/
/* Add station to station queue.                                         */

PUBLIC
void st_add_station (st_name, component, channel)
char st_name[];
char component;
unsigned int channel;
{
   Q_STATION far * qst;
   Q_TYPE type;
   int atod_gain;

   qst = (Q_STATION far *) _fmalloc (sizeof(Q_STATION));
   if (qst == NULL) er_abort (ST_NO_STORAGE);
   suds_initialize_tag (STATIONCOMP, &qst->structtag);

   atod_gain = dt_get_channel_gain(channel);

   /* Get suds structure from master station list, if possible */
   if (! in_masterlist (netwname, st_name, component, &qst->info)) {
      suds_initialize (STATIONCOMP, &qst->info);
      u_strncpy (qst->info.sc_name.network, ((char far *)netwname), 4);
      u_strncpy (qst->info.sc_name.st_name, ((char far *)st_name), 5);
      qst->info.sc_name.component = component;
      qst->info.sensor_type = 't';
      qst->info.polarity    = 'n';
      qst->info.effective = u_timestamp();
      qst->info.max_gain = atod_gain;
      qst->in_masterlist = FALSE;
   } else {
      qst->info.max_gain *= atod_gain;
      qst->in_masterlist = TRUE;
      mean_lat += qst->info.st_lat;
      mean_lon += qst->info.st_long;
      n_stations++;
   }

   qst->info.channel = channel;
   qst->info.atod_gain = atod_gain;

   qst->info.data_type   = dt_get_data_type ();
   qst->info.data_units  = dt_get_data_units ();
   qst->info.clip_value = ((float)dt_get_clip_value ()) - dt_get_dc_offset ();
   qst->info.con_mvolts = (dt_get_input_range () / qst->info.max_gain *
                           1000.00) / (qst->info.clip_value * 2.00);
   type.station = qst;
   q_enqueue (&station_queue, type);
}


/*=======================================================================*
 *                          st_build_textline                            *
 *=======================================================================*/
/* Build a textline for the mscreen module.                              */

PUBLIC
int st_build_textline (channel, textline)
unsigned int channel;
char * textline;
{
   Q_LINK * head;
   char st_name[5];
   int j;

   head = station_queue.head;
   for (head = station_queue.head; head != NULL; head = head->next)
      if (head->type.station->info.channel == channel) {
         u_strncpy (((char far *)st_name), head->type.station->info.sc_name.st_name, 4);
         j = sprintf (textline, "%3d %-4.4s", channel, st_name);
         return (j);
      }
   return ( 0);
}

/*=======================================================================*
 *                           st_get_authority                            *
 *=======================================================================*/
/* Get organiziation processing the data.                                */

PUBLIC
int st_get_authority ()
{
   return (authority);
}

/*=======================================================================*
 *                         st_get_head_station                           *
 *=======================================================================*/
/* Get station at head of station queue.                                 */

PUBLIC
Q_STATION far * st_get_head_station ()
{
   head_ptr = station_queue.head;
   if (head_ptr != NULL)
      return (head_ptr->type.station);
   else return (NULL);
}

/*=======================================================================*
 *                        st_get_next_station                            *
 *=======================================================================*/
/* Get next station from station queue.                                  */

PUBLIC
Q_STATION far * st_get_next_station ()
{
   head_ptr = head_ptr->next;
   if (head_ptr != NULL)
      return (head_ptr->type.station);
   else return (NULL);
}

/*=======================================================================*
 *                            st_get_netwname                            *
 *=======================================================================*/
/* Get network name.                                                     */

PUBLIC
char * st_get_netwname ()
{
   return (netwname);
}

/*=======================================================================*
 *                            st_get_station                             *
 *=======================================================================*/
/* Get station from station queue.                                       */

PUBLIC
Q_STATION far * st_get_station (channel)
unsigned int channel;
{
   Q_LINK * head;

   head = station_queue.head;
   while (head != NULL) {
      if (head->type.station->info.channel == channel)
         return (head->type.station);
      head = head->next;
   }
   return ( NULL);
}

/*=======================================================================*
 *                         st_initialize_params                          *
 *=======================================================================*/
/* Initialize key parameters.                                            */

PUBLIC
void st_initialize_params ()
{
   authority = AUTHORITY;
   mean_lat = mean_lon = 0.0;
   n_stations = 0;
   strcpy (netwname, NETWORK_NAME);
   q_initialize (&station_queue);

   if (station_fd = open_file (STATION_FILENAME, READ))
      station_size = (unsigned int) (filelength (station_fd) /
                      (sizeof(SUDS_STRUCTTAG) + sizeof(SUDS_STATIONCOMP)));
   else er_abort (ST_INVALID_FILE);
}

/*=======================================================================*
 *                        st_initialize_stations                         *
 *=======================================================================*/
/* Initialize station list.                                              */

PUBLIC
void st_initialize_stations ()
{
   Q_LINK * link_ptr;
   LL_XY xy;

   if (n_stations == 0) return;
   mean_lat /= n_stations;
   mean_lon /= n_stations;

   mean_latlon = ll_fold (mean_lat, -mean_lon);
   for (link_ptr = station_queue.head; link_ptr != NULL; link_ptr = link_ptr->next) {
      if (link_ptr->type.station->in_masterlist) {
         link_ptr->type.station->mean_latlon = mean_latlon;
         xy = ll_to_xy (link_ptr->type.station->info.st_lat,
                        -link_ptr->type.station->info.st_long,
                        mean_latlon);
         link_ptr->type.station->x = xy.x;
         link_ptr->type.station->y = xy.y;
         link_ptr->type.station->xsq = xy.x * xy.x;
         link_ptr->type.station->ysq = xy.y * xy.y;
      }
   }
   close_file (station_fd);
}

/*=======================================================================*
 *                           st_set_authority                            *
 *=======================================================================*/
/* Set authority code.                                                   */

PUBLIC
void st_set_authority (code)
int code;
{
   authority = code;
}

/*=======================================================================*
 *                              st_set_gain                              *
 *=======================================================================*/
/* Set gain for particular channel.                                      */

PUBLIC
void st_set_gain (channel, gain)
unsigned int channel;
unsigned int gain;
{
   Q_LINK * head;

   head = station_queue.head;
   while (head != NULL) {
      if (head->type.station->info.channel == channel) {
         head->type.station->info.con_mvolts *= head->type.station->info.max_gain;
         head->type.station->info.max_gain /= head->type.station->info.atod_gain;
         head->type.station->info.max_gain *= gain;
         head->type.station->info.con_mvolts /= head->type.station->info.max_gain;
         head->type.station->info.atod_gain = gain;
         return;
      }
      head = head->next;
   }
}

/*=======================================================================*
 *                            st_set_netwname                            *
 *=======================================================================*/
/* Set network name.                                                     */

PUBLIC
void st_set_netwname (nwn)
char nwn[];
{
   strncpy (netwname, nwn, 4);
   t_set_netwname (nwn);
}
